<?php
/**
 * Luminova Framework Method Class Route Error Attribute
 *
 * @package Luminova
 * @author Ujah Chigozie Peter
 * @copyright (c) Nanoblock Technology Ltd
 * @license See LICENSE file
 * @link https://luminova.ng
 */
namespace Luminova\Attributes;

use \Attribute;
use \Luminova\Exceptions\RouterException;

/**
 * Attribute to define an error handler method for specific HTTP status codes.
 * 
 * This attribute is applied to controller methods to mark them as handlers
 * for one or more HTTP error status codes. When `$router->trigger(<status>)` 
 * is called or an error occurs during routing, the matching controller method 
 * will be executed automatically.
 * 
 * You can specify a single status code (e.g. `404`) or an array of codes 
 * (e.g. `[401, 403, 500]`). The status codes must be valid HTTP codes 
 * (100–599), otherwise a `RouterException` is thrown during attribute 
 * initialization.
 *
 * ### Parameters
 * - **$status** (`int|array`)  
 *   The HTTP status code or list of codes that this method will handle.
 *
 * ### Exceptions
 * - **RouterException**  
 *   Thrown if a provided status code is invalid (not numeric or outside 100–599).
 *
 * ### Example Usage:
 * ```php
 * use Luminova\Attributes\Status;
 * use Luminova\Base\Controller;
 * 
 * class ErrorController extends Controller 
 * {
 *     #[Status(404)]
 *     public function onNotFound(): void {
 *         // handle 404 not found
 *     }
 * 
 *     #[Status([401, 403])]
 *     public function onAuthErrors(): void {
 *         // handle unauthorized/forbidden
 *     }
 * }
 * ```
 */
#[Attribute(Attribute::TARGET_METHOD)]
final class Status
{
    public function __construct(public int|array $status = 404)
    {
        $this->assert();
    }

    private function isValid(int $code): bool 
    {
        return $code >= 100 && $code <= 599;
    }

    private function assertInt(mixed $code): void 
    {
        if (!is_int($code)) {
            return;
        }

        throw new RouterException(
            "Status must be an integer or an array of integers."
        );
    }

    private function assert(): void 
    {
        if (is_int($this->status)) {
            if (!$this->isValid($this->status)) {
                throw new RouterException(
                    "Invalid HTTP status code: {$this->status}. Must be between 100 and 599."
                );
            }
            return;

        }
        
        foreach ($this->status as $code) {
            $this->assertInt($code);

            if (!$this->isValid((int) $code)) {
                throw new RouterException(
                    "Invalid HTTP status code in array: {$code}. Must be integers between 100 and 599."
                );
            }
        }
    }
}